#include "Allocator.h"
#include <malloc.h>

#ifdef WIN32
#include <crtdbg.h>
#else
#define _RPT0(a, b)
#endif

#define	OBJ_AREA_SIZE 2048

//PoolAllocatorSet    gAllocators;

//
//	Object-based Allocation
//

bool	debug_allocation = false;
bool    trace_allocation = false;
long	sizeAllocated;
static char trace_alloc_buff[256];
int     trace_alloc_counter;
int     trace_delete_counter;

struct AllocItem {
	AllocItem *next;
	void	*obj;
};
struct AllocList {
	int	objSize;
	AllocList *next;
	AllocItem *freeItems;
        AllocItem *allocatedItems;
        int     nFree, nAllocated;
} *allocList;

#include <windows.h>
#include <psapi.h>

Char    used_memory[128];
unsigned long lastWorkingSetSize;
unsigned long incrSize;

unsigned long    checkMemory()
{
#if 01
        HANDLE proc = GetCurrentProcess();
        PROCESS_MEMORY_COUNTERS counters;
        GetProcessMemoryInfo(proc, &counters, sizeof(counters));
        lastWorkingSetSize = counters.WorkingSetSize;
#endif
        wxSprintf(used_memory, "Mem.: %ld kB", (unsigned long)(lastWorkingSetSize / 1024));
        return lastWorkingSetSize;
}

void	*operator new(size_t size)
{
        void    *ptr;

	if(!debug_allocation)
        {
	    ptr = calloc(size, 1);
            if(trace_allocation)
            {
                sprintf(trace_alloc_buff, "%7d: new [%d] = %p\n", trace_alloc_counter++, size, ptr);
                _RPT0(_CRT_WARN, trace_alloc_buff);
            }
            return ptr;
        }

	// find a list of free objects of the right size
	AllocList *list;
	for(list = allocList; list; list = list->next)
	    if(size == list->objSize)
		break;

	if(!list)
	{
	    list = (AllocList *)malloc(sizeof(AllocList));
	    list->objSize = size;
            list->freeItems = list->allocatedItems = 0;
            list->nFree = list->nAllocated = 0;
	    list->next = allocList;
	    allocList = list;
	}

	// get the item from the list
	AllocItem *item = list->freeItems;
	if(!item)
	{
	    item = (AllocItem *)malloc(sizeof(AllocItem));
            item->next = list->allocatedItems;
            list->allocatedItems = item;
	    item->obj = calloc(size, 1);
	}
	else
	{
	    list->freeItems = item->next;
            item->next = list->allocatedItems;
            list->allocatedItems = item;
            --list->nFree;
	}
	sizeAllocated += size;
        ++list->nAllocated;
        return item->obj;
}


void	operator delete(void *obj)
{
        if(!obj)
	    return;
	if(!debug_allocation)
	{
            if(trace_allocation)
            {
                sprintf(trace_alloc_buff, "%7d: delete %p\n", trace_delete_counter++, obj);
                _RPT0(_CRT_WARN, trace_alloc_buff);
            }
	    free(obj);
	    return;
	}
        AllocList   *list;
        AllocItem   *item, *prev;
        for(list = allocList; list; list = list->next)
        {
            prev = 0;
            for(item = list->allocatedItems; item; prev = item, item = item->next)
            {
                if(item->obj != obj)
                    continue;
                if(prev)
                    prev->next = item->next;
                else
                    list->allocatedItems = item->next;
                item->next = list->freeItems;
                list->freeItems = item;
                sizeAllocated -= list->objSize;
                --list->nAllocated;
                ++list->nFree;
                return;
            }
        }
	fprintf(stderr, "panic: deleting non-allocated object!\n");
}




void	*operator new (size_t s, PoolAllocator *p)
{
	PoolAllocator *a = (PoolAllocator *)p;
	return p->Allocate();
}


void	operator delete (void *obj, PoolAllocator *p)
{
	// nothing to do, since we never call this
}



PoolAllocator::PoolAllocator(int size)
{
	_pages = 0;
	_objSize = size;
        _nFreePages = 0;
        _nAllocatedPages = 0;
}


PoolAllocator::~PoolAllocator()
{
	Release();
}


void	*PoolAllocator::Allocate()
{
	PoolAllocatorPage *page = _pages;
	void	*objptr;

	if(!page || page->_nextOffset + _objSize >= page->_maxOffset) {
            if(_free_pages) {
                page = _free_pages;
                _free_pages = page->_next;
                --_nFreePages;
                ++_nAllocatedPages;
            } else {
	        page = (PoolAllocatorPage *)calloc(sizeof(PoolAllocatorPage), 1);
	        page->_maxOffset = (OBJ_AREA_SIZE / _objSize) * _objSize;
	        page->_base = (char *)malloc(OBJ_AREA_SIZE);
                ++_nAllocatedPages;
            }
            page->_next = _pages;
            _pages = page;
	    page->_nextOffset = 0;
	}
	objptr = (void *)(page->_base + page->_nextOffset);
	page->_nextOffset += _objSize;
	return objptr;
}


void    PoolAllocator::Reset()
{
        while(_pages) {
            PoolAllocatorPage *next = _pages->_next;
            _pages->_next = _free_pages;
            _free_pages = _pages;
            _pages = next;
            --_nAllocatedPages;
            ++_nFreePages;
        }
}


void	PoolAllocator::Release()
{
	PoolAllocatorPage *page;

        Reset(); // move any allocated pages to the free pages list
	while((page = _free_pages)) {
	    free(page->_base);
	    _free_pages = page->_next;
	    free(page);
	}
        // now both lists should be empty
}


unsigned long PoolAllocator::GetAllocatedSize() const
{
	PoolAllocatorPage *page;
	unsigned long total = 0;

	for(page = _pages; page; page = page->_next)
	    total += page->_nextOffset;
	return total;
}

StringPool::StringPool()
{
        _pages = (Char **)malloc(sizeof(Char *));
        _pages[0] = (Char *)malloc(STRING_PAGE_SIZE);
        _nPages = 1;
        _curPage = 0;
        _pagePos = 0;
}

StringPool::~StringPool()
{
        int     x;
        for(x = 0; x < _nPages; ++x)
            free(_pages[x]);
        free(_pages);
        _pages = 0;
        _nPages = 0;
        _curPage = 0;
        _pagePos = 0;
}

void    StringPool::Reset()
{
        _curPage = 0;
        _pagePos = 0;
}

Char    *StringPool::Allocate(const Char *str)
{
        Char    *ptr;
        int neededSize = (wxStrlen(str) + 1) * sizeof(Char);
        if(_pagePos + neededSize >= STRING_PAGE_SIZE) {
            ++_curPage;
            _pagePos = 0;
            if(_curPage >= _nPages) {
                ++_nPages;
                _pages = (Char **)realloc(_pages, sizeof(Char **) * _nPages);
                _pages[_nPages - 1] = (Char *)malloc(STRING_PAGE_SIZE);
            }
        }
        ptr = _pages[_curPage] + _pagePos;
        wxStrcpy(ptr, str);
        _pagePos += neededSize;
        return ptr;
}
