/*	Shape.cpp - Created by Giampiero Caprino

This file is part of Train Director 3

Train Director is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; using exclusively version 2.
It is expressly forbidden the use of higher versions of the GNU
General Public License.

Train Director is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Train Director; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.
*/

#include "Traindir3.h"
#include "trsim.h"
#include "ui.h"

TrackShape::TrackShape()
{
	memset(_out, (int)CD_NONE, sizeof(_out));
}

TrackShape::~TrackShape()
{
}

//	X and Y offsets from current element coordinate
//	for the specified (compass) direction

static	int dx[] = {
	0,  // CD_NONE
	-1, // CD_SW
	0,  // CD_S
	1,  // CD_SE
	-1, // CD_W
	0,  // CD_SPOT
	1,  // CD_E
	-1, // CD_NW
	0,  // CD_N
	1   // CD_NE
};
static	int dy[] = {
    	0,  // CD_NONE
	1,  // CD_SW
	1,  // CD_S
	1,  // CD_SE
	0,  // CD_W
	0,  // CD_SPOT
	0,  // CD_E
	-1, // CD_NW
	-1, // CD_N
	-1  // CD_NE
};

static	CompassDir opposing_directions[] = {
	CD_NONE,// CD_NONE
	CD_NE,  // CD_SW
	CD_N,   // CD_S
	CD_NW,  // CD_SE
	CD_E,   // CD_W
	CD_SPOT,// CD_SPOT
	CD_W,   // CD_E
	CD_SE,  // CD_NW
	CD_S,   // CD_N
	CD_SW   // CD_NE
};
static  const   char *direction_names[] = {
        wxT("None"),
        wxT("SW"),
        wxT("S"),
        wxT("SE"),
        wxT("W"),
        wxT("Spot"),
        wxT("E"),
        wxT("NW"),
        wxT("N"),
        wxT("NE")
};

// These don't seem to be used at all!
CompassDir	n_s_segs[] = { CD_N, CD_S, CD_NONE };
CompassDir	sw_n_segs[] = { CD_SW, CD_N, CD_NONE };
CompassDir	nw_s_segs[] = { CD_NW, CD_S, CD_NONE };
CompassDir	w_e_segs[] = { CD_W, CD_E, CD_NONE };
CompassDir	nw_e_segs[] = { CD_NW, CD_E, CD_NONE };
CompassDir	sw_e_segs[] = { CD_SW, CD_E, CD_NONE };
CompassDir	w_ne_segs[] = { CD_W, CD_NE, CD_NONE };
CompassDir	w_se_segs[] = { CD_W, CD_SE, CD_NONE };
CompassDir	nw_se_segs[] = { CD_NW, CD_SE, CD_NONE };
CompassDir	sw_ne_segs[] = { CD_SW, CD_NE, CD_NONE };
CompassDir	ne_s_segs[] = { CD_NE, CD_S, CD_NONE };
CompassDir	se_n_segs[] = { CD_SE, CD_N, CD_NONE };
CompassDir	itin_segs[] = { CD_NW, CD_SW, CD_NE, CD_SE, CD_W, CD_E, CD_NONE };

//  Track elements

TrackShape  w_e;
TrackShape  n_s;
TrackShape  nw_se;
TrackShape  sw_ne;
TrackShape  nw_e;
TrackShape  sw_e;
TrackShape  w_ne;
TrackShape  w_se;
TrackShape  nw_s;
TrackShape  ne_s;
TrackShape  n_sw;
TrackShape  n_se;

//  level-crossing (diamonds) elements

TrackShape  diamond_x;
TrackShape  diamond_plus;
TrackShape  diamond_h_sw_ne;
TrackShape  diamond_h_nw_se;
TrackShape  diamond_v_sw_ne;
TrackShape  diamond_v_nw_se;

TrackShape *track_shape_map[N_NW_S_SE + 1];  // see trkdir to TrackShape

//      English switches have special paths for their thrown states

TrackShape  main_we_sw_ne, branch_we_sw_ne;
TrackShape  main_we_nw_se, branch_we_nw_se;
TrackShape  main_ns_nw_se, branch_ns_nw_se;
TrackShape  main_ns_sw_ne, branch_ns_sw_ne;

struct edittools tooltbltracks[] = {   /* used when screen is 800x600 */
	{ TEXT, 0, 0, 0 },
        { TRACK, TRK_N_S, 0, 1, &n_s },
	{ TRACK, W_E, 1, 1, &w_e },
	{ TRACK, NW_SE, 2, 1, &nw_se },
	{ TRACK, SW_NE, 3, 1, &sw_ne },
        { TRACK, W_NE, 4, 1, &w_ne },
	{ TRACK, W_SE, 5, 1, &w_se },
	{ TRACK, NW_E, 6, 1, &nw_e },
        { TRACK, SW_E, 7, 1, &sw_e },
        { TRACK, NW_S, 8, 1, &nw_s },
        { TRACK, SW_N, 9, 1, &n_sw },
        { TRACK, NE_S, 10, 1, &ne_s },
	{ TRACK, SE_N, 11, 1, &n_se },
        { TRACK, XH_NW_SE, 12, 1, &diamond_h_nw_se },
        { TRACK, XH_SW_NE, 13, 1, &diamond_h_sw_ne },
        { TRACK, X_X, 14, 1, &diamond_x },
        { TRACK, X_PLUS, 15, 1, &diamond_plus },
        { TRACK, N_NE_S_SW, 16, 1, &diamond_v_sw_ne },	// no switch  / |
        { TRACK, N_NW_S_SE, 17, 1, &diamond_v_nw_se },	// no switch  \ |
	{ -1 }
};
struct edittools tooltblswitches[] = {
	{ TEXT, 0, 0, 0 },
	{ SWITCH, 0, 0, 1, &w_e, &w_ne },
        { SWITCH, 1, 1, 1, &w_e, &nw_e },
        { SWITCH, 2, 2, 1, &w_e, &w_se },
        { SWITCH, 3, 3, 1, &w_e, &sw_e },
        { SWITCH, 4, 4, 1, &sw_ne, &sw_e },
        { SWITCH, 5, 5, 1, &sw_ne, &w_ne },
        { SWITCH, 6, 6, 1, &nw_se, &nw_e },
        { SWITCH, 7, 7, 1, &nw_se, &w_se },
        { SWITCH, 8, 8, 1, &main_we_sw_ne, &branch_we_sw_ne },
        { SWITCH, 9, 9, 1, &main_we_nw_se, &branch_we_nw_se },
        { SWITCH, 10, 10, 1, &w_ne, &w_se }, // TODO: check which is main and which is branch
        { SWITCH, 11, 11, 1, &nw_e, &sw_e }, // TODO: check which is main and which is branch

        { SWITCH, 12, 12, 1, &n_s, &n_sw },	    /* vertical switches */
        { SWITCH, 13, 13, 1, &n_s, &n_se },
        { SWITCH, 14, 14, 1, &n_s, &nw_s },
        { SWITCH, 15, 15, 1, &n_s, &ne_s },
        { SWITCH, 16, 16, 1, &main_ns_sw_ne, &branch_ns_sw_ne },
        { SWITCH, 17, 17, 1, &main_ns_nw_se, &branch_ns_nw_se },
        { SWITCH, 18, 18, 1, &sw_ne, &n_sw },
        { SWITCH, 19, 19, 1, &sw_ne, &ne_s },
        { SWITCH, 20, 20, 1, &nw_se, &n_se },
        { SWITCH, 21, 21, 1, &nw_se, &nw_s },
        { SWITCH, 22, 22, 1, &ne_s, &nw_s }, // TODO: check which is main and which is branch
        { SWITCH, 23, 23, 1, &n_se, &n_sw }, // -TODO: check which is main and which is branch
	{ -1 }
};
struct edittools tooltblsignals[] = {
	{ TEXT, 0, 0, 0 },
	{ TSIGNAL, 0, 0, 1 },
	{ TSIGNAL, 1, 1, 1 },
	{ TSIGNAL, 2, 2, 1 },
	{ TSIGNAL, 3, 3, 1 },
	{ TSIGNAL, S_N, 4, 1 },
	{ TSIGNAL, N_S, 5, 1 },
	{ TSIGNAL, signal_NORTH_FLEETED, 6, 1 },
	{ TSIGNAL, signal_SOUTH_FLEETED, 7, 1 },
	{ -1 }
};

trkdir  compass_to_trkdir[CD_MAX] = {
	W_E, // CD_NONE = 0,// impossible
	E_W, // CD_SW = 1,
	N_S, // CD_S = 2,
	W_E, // CD_SE = 3,
	E_W, // CD_W = 4,
	W_E, // CD_SPOT = 5, // impossible
	W_E, // CD_E = 6,
	E_W, // CD_NW = 7,
	S_N, // CD_N = 8,
	W_E  // CD_NE = 9,
};

void	init_track_shapes()
{
	// set outgoing path directions for valid incoming directions

	track_shape_map[W_E] = &w_e;
	track_shape_map[E_W] = &w_e;	// :-(
	w_e._out[CD_E] = CD_E;		    // straight line elements
	w_e._out[CD_W] = CD_W;

	track_shape_map[TRK_N_S] = &n_s;
	track_shape_map[N_S] = &n_s;	// :-(
	track_shape_map[S_N] = &n_s;	// :-(
	n_s._out[CD_N] = CD_N;
	n_s._out[CD_S] = CD_S;

	track_shape_map[NW_SE] = &nw_se;
	nw_se._out[CD_SE] = CD_SE;
	nw_se._out[CD_NW] = CD_NW;

	track_shape_map[SW_NE] = &sw_ne;
	sw_ne._out[CD_NE] = CD_NE;
	sw_ne._out[CD_SW] = CD_SW;

	track_shape_map[NW_E] = &nw_e;
	nw_e._out[CD_SE] = CD_E;	    // horizondal bends elements
	nw_e._out[CD_W] = CD_NW;

	track_shape_map[W_SE] = &w_se;
	sw_e._out[CD_NE] = CD_E;
	sw_e._out[CD_W] = CD_SW;

	track_shape_map[W_NE] = &w_ne;
	w_ne._out[CD_E] = CD_NE;
	w_ne._out[CD_SW] = CD_W;

	track_shape_map[W_SE] = &w_se;
	w_se._out[CD_E] = CD_SE;
	w_se._out[CD_NW] = CD_W;

	track_shape_map[NW_S] = &nw_s;
	nw_s._out[CD_SE] = CD_S;	    // vertical bends elements
	nw_s._out[CD_N] = CD_NW;

	track_shape_map[NE_S] = &ne_s;
	ne_s._out[CD_SW] = CD_S;
	ne_s._out[CD_N] = CD_NE;

	track_shape_map[SW_N] = &n_sw;
	n_sw._out[CD_S] = CD_SW;
	n_sw._out[CD_NE] = CD_N;

	track_shape_map[SE_N] = &n_se;
	n_se._out[CD_S] = CD_SE;
	n_se._out[CD_NW] = CD_N;

	track_shape_map[X_X] = &diamond_x;
	diamond_x._out[CD_SE] = CD_SE;
	diamond_x._out[CD_NE] = CD_NE;
	diamond_x._out[CD_SW] = CD_SW;
	diamond_x._out[CD_NW] = CD_NW;

	track_shape_map[X_PLUS] = &diamond_plus;
	diamond_plus._out[CD_E] = CD_E;
	diamond_plus._out[CD_W] = CD_W;
	diamond_plus._out[CD_S] = CD_S;
	diamond_plus._out[CD_N] = CD_N;

	track_shape_map[XH_SW_NE] = &diamond_h_sw_ne;
	diamond_h_sw_ne._out[CD_E] = CD_E;
	diamond_h_sw_ne._out[CD_W] = CD_W;
	diamond_h_sw_ne._out[CD_NE] = CD_NE;
	diamond_h_sw_ne._out[CD_SW] = CD_SW;
	
	track_shape_map[XH_NW_SE] = &diamond_h_nw_se;
	diamond_h_nw_se._out[CD_E] = CD_E;
	diamond_h_nw_se._out[CD_W] = CD_W;
	diamond_h_nw_se._out[CD_SE] = CD_SE;
	diamond_h_nw_se._out[CD_NW] = CD_NW;

	track_shape_map[N_NE_S_SW] = &diamond_v_sw_ne;
	diamond_v_sw_ne._out[CD_N] = CD_N;
	diamond_v_sw_ne._out[CD_S] = CD_S;
	diamond_v_sw_ne._out[CD_NE] = CD_NE;
	diamond_v_sw_ne._out[CD_SW] = CD_SW;
	
	track_shape_map[N_NW_S_SE] = &diamond_v_nw_se;
	diamond_v_nw_se._out[CD_N] = CD_N;
	diamond_v_nw_se._out[CD_S] = CD_S;
	diamond_v_nw_se._out[CD_SE] = CD_SE;
	diamond_v_nw_se._out[CD_NW] = CD_NW;

	// Switches
        //
        //  Only four-way switches require special setup
        //  The regular three-way switches can be handled
        //  by standard track shapes (one for main, one for branch line)

        main_we_sw_ne._out[CD_E] = CD_E;
        main_we_sw_ne._out[CD_W] = CD_W;
        main_we_sw_ne._out[CD_NE] = CD_NE;
        main_we_sw_ne._out[CD_SW] = CD_SW;

        branch_we_sw_ne._out[CD_E] = CD_NE;
        branch_we_sw_ne._out[CD_W] = CD_SW;
        branch_we_sw_ne._out[CD_SW] = CD_W;
        branch_we_sw_ne._out[CD_NE] = CD_E;

        main_we_nw_se._out[CD_E] = CD_E;
        main_we_nw_se._out[CD_W] = CD_W;
        main_we_nw_se._out[CD_SE] = CD_SE;
        main_we_nw_se._out[CD_NW] = CD_NW;

        branch_we_nw_se._out[CD_E] = CD_SE;
        branch_we_nw_se._out[CD_W] = CD_NW;
        branch_we_nw_se._out[CD_SE] = CD_E;
        branch_we_nw_se._out[CD_NW] = CD_W;

        main_ns_nw_se._out[CD_N] = CD_N;
        main_ns_nw_se._out[CD_S] = CD_S;
        main_ns_nw_se._out[CD_SE] = CD_SE;
        main_ns_nw_se._out[CD_NW] = CD_NW;

        branch_ns_nw_se._out[CD_N] = CD_NW;
        branch_ns_nw_se._out[CD_S] = CD_SE;
        branch_ns_nw_se._out[CD_NW] = CD_N;
        branch_ns_nw_se._out[CD_SE] = CD_S;

        main_ns_sw_ne._out[CD_N] = CD_N;
        main_ns_sw_ne._out[CD_S] = CD_S;
        main_ns_sw_ne._out[CD_SW] = CD_SW;
        main_ns_sw_ne._out[CD_NE] = CD_NE;

        branch_ns_sw_ne._out[CD_N] = CD_NE;
        branch_ns_sw_ne._out[CD_S] = CD_SW;
        branch_ns_sw_ne._out[CD_NE] = CD_N;
        branch_ns_sw_ne._out[CD_SW] = CD_S;

}

// old directions are ambiguous
// E_W may point to a w_se or w_ne or w_e element
// since we're entering the block, we must pick a valid
// CompassDir for walk_path() to work. There should
// only be one valid direction per each trackShape
// from the desired ambiguousdirection

CompassDir TrackShape::GetValidEntryDirection(CompassDir ambiguous)
{
        switch(ambiguous) {
        case CD_W:
            if(_out[CD_W]) return CD_W;
            if(_out[CD_NW]) return CD_NW;
            if(_out[CD_SW]) return CD_SW;
            break;

        case CD_E:
            if(_out[CD_E]) return CD_E;
            if(_out[CD_NE]) return CD_NE;
            if(_out[CD_SE]) return CD_SE;
            break;

        case CD_N:
            if(_out[CD_N]) return CD_N;
            if(_out[CD_NW]) return CD_NW;
            if(_out[CD_NE]) return CD_NE;
            break;

        case CD_S:
            if(_out[CD_S]) return CD_S;
            if(_out[CD_SW]) return CD_SW;
            if(_out[CD_SE]) return CD_SE;
            break;
        }
        return ambiguous; // TODO: should we return CD_NONE?
}

trkdir TrackShape::CompassToTrkdir(CompassDir compass)
{
        return compass_to_trkdir[compass];
}

CompassDir TrackShape::TrkdirToCompass(trkdir tdir)
{
        CompassDir cdir;

        switch(tdir) {
        case E_W:
            cdir = CD_W;
            break;
        case N_S:
            cdir = CD_S;
            break;
        case S_N:
            cdir = CD_N;
            break;
        case W_E:
        default:
            cdir = CD_E;
        }
        return cdir;
}

CompassDir TrackShape::GetOppositeDir(const CompassDir dir)
{
        return opposing_directions[dir];
}

void    TrackShape::GetOutDirNames(const Char *names[])
{
        int out_idx = 0;
        for(int i = 0; i < CD_MAX; ++i)
            if(this->_out[i] != CD_NONE) {
                names[out_idx++] = direction_names[this->_out[i]];
            }
        names[out_idx] = 0;
}

CompassDir TrackShape::GetDirFromName(const Char *name)
{
        int     i;
        for(i = 0; i < CD_MAX; ++i)
            if(!wxStrcmp(name, direction_names[i]))
                return (CompassDir)i;
        return CD_NONE;
}

void    set_track_shape(Track *t)
{
        struct edittools *tbl = tooltbltracks;
        if(t->type == SWITCH) {
            tbl = tooltblswitches;
        } else {
            // TODO: handle signals
        }
        int i;
        for(i = 1; tbl[i].type != -1; ++i)
            if(tbl[i].direction == t->direction) {
                t->_trackShape = tbl[i].shape;
                t->_branchShape = tbl[i].branchShape;
                break;
            }
}

int	walk_track(TrackBase *trk, Coord& next_xy, CompassDir& dir)
{
	TrackShape* shape = trk->_trackShape;
        if(!shape) // TEXT elements have no shape
            return 0;
        if(trk->type == SWITCH && trk->switched)
            shape = trk->_branchShape;

	CompassDir fromDir = dir;
	CompassDir toDir = shape->_out[fromDir];
	if(toDir == CD_NONE)
	    return 0;
	dir = toDir;

        // check if next_xy is from a link
        // this must be after we decided the outbound direction
        // from the current track to correctly enter the linked track

        if(trk->type == TRACK) { // not for switches, since their links work differently
            switch(dir) {
            case CD_N:
                if(fromDir != CD_N) {
                    break; // handle link only for vertical-to-vertical (no curves)
                }
                if(trk->wlinkx && trk->wlinky) {
                    // some scenario (VeneziaMestre) use a track linked
                    // to itself to avoid putting a signal at the end of a track spur
                    if(trk->wlinkx == trk->x && trk->wlinky == trk->y)
                        return 0;
	            next_xy.x = trk->wlinkx;
	            next_xy.y = trk->wlinky;
                    trk = findTrack(next_xy.x, next_xy.y);
                    CompassDir ndir = trk->_trackShape->GetValidEntryDirection(CD_N);
                    dir = ndir;
	            return 1;
                }
                break;
            case CD_W:
            case CD_NW:
            case CD_SW:
                if(trk->wlinkx && trk->wlinky) {
                    // some scenario (VeneziaMestre) use a track linked
                    // to itself to avoid putting a signal at the end of a track spur
                    if(trk->wlinkx == trk->x && trk->wlinky == trk->y)
                        return 0;
	            next_xy.x = trk->wlinkx;
	            next_xy.y = trk->wlinky;
                    trk = findTrack(next_xy.x, next_xy.y);
                    CompassDir ndir = trk->_trackShape->GetValidEntryDirection(CD_W);
                    dir = ndir;
	            return 1;
                }
                break;
            case CD_S:
                if(fromDir != CD_S) {
                    break; // handle link only for vertical-to-vertical (no curves)
                }
                if(trk->elinkx && trk->elinky) {
                    // some scenario (VeneziaMestre) use a track linked
                    // to itself to avoid putting a signal at the end of a track spur
                    if(trk->elinkx == trk->x && trk->elinky == trk->y)
                        return 0;
	            next_xy.x = trk->elinkx;
	            next_xy.y = trk->elinky;
                    trk = findTrack(next_xy.x, next_xy.y);
                    CompassDir ndir = trk->_trackShape->GetValidEntryDirection(CD_S);
                    dir = ndir;
	            return 1;
                }
                break;
            case CD_E:
            case CD_NE:
            case CD_SE:
                if(trk->elinkx && trk->elinky) {
                    // some scenario (VeneziaMestre) use a track linked
                    // to itself to avoid putting a signal at the end of a track spur
                    if(trk->elinkx == trk->x && trk->elinky == trk->y)
                        return 0;
	            next_xy.x = trk->elinkx;
	            next_xy.y = trk->elinky;
                    trk = findTrack(next_xy.x, next_xy.y);
                    CompassDir ndir = trk->_trackShape->GetValidEntryDirection(CD_E);
                    dir = ndir;
	            return 1;
                }
                break;
	    }
        }
	next_xy.x = trk->x + dx[dir];
	next_xy.y = trk->y + dy[dir];
	return 1;
}
